﻿/*
* Copyright (c) Contributors, http://opensimulator.org/
* See CONTRIBUTORS.TXT for a full list of copyright holders.
*
* Redistribution and use in source and binary forms, with or without
* modification, are permitted provided that the following conditions are met:
*     * Redistributions of source code must retain the above copyright
*       notice, this list of conditions and the following disclaimer.
*     * Redistributions in binary form must reproduce the above copyright
*       notice, this list of conditions and the following disclaimer in the
*       documentation and/or other materials provided with the distribution.
*     * Neither the name of the OpenSim Project nor the
*       names of its contributors may be used to endorse or promote products
*       derived from this software without specific prior written permission.
*
* THIS SOFTWARE IS PROVIDED BY THE DEVELOPERS ``AS IS'' AND ANY
* EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
* WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
* DISCLAIMED. IN NO EVENT SHALL THE CONTRIBUTORS BE LIABLE FOR ANY
* DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
* (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
* LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
* ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
* (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
* SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
* 
*/
using System;
using System.Collections.Generic;
using System.Data;
using System.Text.RegularExpressions;
using libsecondlife;
using NHibernate;
using NHibernate.Expression;
using OpenSim.Framework;

namespace OpenSim.Framework.Data.Hibernate
{
    /// <summary>
    /// A database interface class to a user profile storage system
    /// </summary>
    public class HibernateUserData : IUserData
    {
        #region IUserData Members

        public void Delete(UserProfileData entity)
        {
            ISession session = NHibernateSessionManager.Instance.GetSession();
            ITransaction tx = session.BeginTransaction();
            session.Delete(entity);
            tx.Commit();
            session.Flush();

        }


        public UserProfileData GetUserByUUID(LLUUID user)
        {
            UserProfileData RetVal = null;
            ISession session = NHibernateSessionManager.Instance.GetSession();
            ITransaction tx = session.BeginTransaction();
            try
            {
                RetVal = session.Load<UserProfileData>(user);
            }
            catch (Exception)
            {
                //retVal will be null;
            }
            tx.Commit();

            return RetVal;
        }

        public UserProfileData GetUserByName(string fname, string lname)
        {
            UserProfileData RetVal = null;

            ISession session = NHibernateSessionManager.Instance.GetSession();

            ITransaction tx = session.BeginTransaction();

            ICriteria criteria = session.CreateCriteria(typeof(UserProfileData));
            criteria.Add(Expression.Eq("username", fname));
            criteria.Add(Expression.Eq("surname", lname));

            RetVal = criteria.UniqueResult<UserProfileData>();

            tx.Commit();

            return RetVal;

        }

        public List<OpenSim.Framework.AvatarPickerAvatar> GeneratePickerResults(LLUUID queryID, string query)
        {

            string[] querysplit;
            querysplit = query.Split(' ');

            int numTerms = querysplit.Length;

            Regex objAlphaNumericPattern = new Regex("[^a-zA-Z0-9]");

            string first = objAlphaNumericPattern.Replace(querysplit[0], String.Empty) + "%";
            string second = first;
            if (numTerms == 2)
            {
                second = objAlphaNumericPattern.Replace(querysplit[1], String.Empty) + "%";
            }

            ISession session = NHibernateSessionManager.Instance.GetSession();

            ITransaction tx = session.BeginTransaction();

            ICriteria criteria = session.CreateCriteria(typeof(UserProfileData));
            IList<UserProfileData> results = null;
            if (numTerms == 2)
            {
                criteria.Add(Expression.Like("username", first));
                criteria.Add(Expression.Like("surname", second));
            }
            else
            {
                Disjunction disjunct = Expression.Disjunction();
                disjunct.Add(Expression.Like("username", first));
                disjunct.Add(Expression.Like("surname", first));
                criteria.Add(disjunct);
            }

            criteria.SetMaxResults(100);

            results = criteria.List<UserProfileData>();
            tx.Commit();

            List<OpenSim.Framework.AvatarPickerAvatar> returnlist = new List<OpenSim.Framework.AvatarPickerAvatar>();

            foreach (UserProfileData profile in results)
            {
                OpenSim.Framework.AvatarPickerAvatar apa = new OpenSim.Framework.AvatarPickerAvatar();
                apa.AvatarID = profile.UUID;
                apa.firstName = profile.username;
                apa.lastName = profile.surname;
                returnlist.Add(apa);
            }

            return returnlist;
        }

        public UserAgentData GetAgentByUUID(LLUUID userID)
        {
            UserProfileData user = GetUserByUUID(userID);
            return user.currentAgent;

        }

        public UserAgentData GetAgentByName(string name)
        {
            //nobody calls this method
            throw new NotImplementedException();
        }

        public UserAgentData GetAgentByName(string fname, string lname)
        {
            UserProfileData user = GetUserByName(fname, lname);
            return user.currentAgent;
        }

        public void StoreWebLoginKey(LLUUID agentID, LLUUID webLoginKey)
        {
            ISession session = NHibernateSessionManager.Instance.GetSession();

            UserProfileData user = GetUserByUUID(agentID);
            user.webLoginKey = webLoginKey;

            ITransaction tx = session.BeginTransaction();

            session.Update(user);

            tx.Commit();



        }

        public void AddNewUserProfile(UserProfileData user)
        {
            ISession session = NHibernateSessionManager.Instance.GetSession();
            ITransaction tx = session.BeginTransaction();
            session.Save(user);
            tx.Commit();
        }

        public bool UpdateUserProfile(UserProfileData user)
        {
            ISession session = NHibernateSessionManager.Instance.GetSession();
            ITransaction tx = session.BeginTransaction();
            session.SaveOrUpdate(user);
            tx.Commit();

            return true;
        }

        public void UpdateUserCurrentRegion(LLUUID avatarid, LLUUID regionuuid)
        {
            //m_log.Info("[USER]: Stub UpdateUserCUrrentRegion called");
        }

        public void AddNewUserAgent(UserAgentData agent)
        {
            ISession session = NHibernateSessionManager.Instance.GetSession();
            ITransaction tx = session.BeginTransaction();
            UserProfileData user = GetUserByUUID(agent.UUID);
            user.currentAgent = agent;

            session.Update(user);
            tx.Commit();
        }

        public void AddNewUserFriend(LLUUID friendlistowner, LLUUID friend, uint perms)
        {
            ISession session = NHibernateSessionManager.Instance.GetSession();
            ITransaction tx = session.BeginTransaction();

            //TODO: this is not OO at all, need to revisit when associations are done
            // do forward and reverse entries
            FriendListItem fli = new FriendListItem();
            fli.FriendListOwner = friendlistowner;
            fli.Friend = friend;
            fli.FriendPerms = perms;
            //fli.datetimestamp = (uint)Utils.Util.UnixTimeSinceEpoch();

            FriendListItem fli2 = new FriendListItem();
            fli2.FriendListOwner = friend;
            fli2.Friend = friendlistowner;
            fli2.FriendPerms = perms;
            //fli2.datetimestamp = (uint)Utils.Utils.UnixTimeSinceEpoch();

            session.Save(fli);
            session.Save(fli2);

            tx.Commit();
        }

        public void RemoveUserFriend(LLUUID friendlistowner, LLUUID friend)
        {
            ISession session = NHibernateSessionManager.Instance.GetSession();
            ITransaction tx = session.BeginTransaction();

            //TODO: this is not OO at all, need to revisit when associations are done
            // do forward and reverse entries
            ICriteria criteria = session.CreateCriteria(typeof(FriendListItem));
            criteria.Add(Expression.Eq("FriendListOwner", friendlistowner));
            criteria.Add(Expression.Eq("Friend", friend));

            FriendListItem fli = criteria.UniqueResult<FriendListItem>();

            if (fli != null)
            {
                session.Delete(fli);
            }


            criteria = session.CreateCriteria(typeof(FriendListItem));
            criteria.Add(Expression.Eq("FriendListOwner", friend));
            criteria.Add(Expression.Eq("Friend", friendlistowner));

            fli = criteria.UniqueResult<FriendListItem>();

            if (fli != null)
            {
                session.Delete(fli);
            }

            tx.Commit();
        }

        public void UpdateUserFriendPerms(LLUUID friendlistowner, LLUUID friend, uint perms)
        {
            ISession session = NHibernateSessionManager.Instance.GetSession();
            ITransaction tx = session.BeginTransaction();

            //TODO: this is not OO at all, need to revisit when associations are done
            // do forward and reverse entries
            ICriteria criteria = session.CreateCriteria(typeof(FriendListItem));
            criteria.Add(Expression.Eq("FriendListOwner", friendlistowner));
            criteria.Add(Expression.Eq("Friend", friend));

            FriendListItem fli = criteria.UniqueResult<FriendListItem>();

            if (fli != null)
            {
                fli.FriendPerms = perms;
                session.Update(fli);
            }


            tx.Commit();
        }

        public List<FriendListItem> GetUserFriendList(LLUUID friendlistowner)
        {
            List<FriendListItem> retVal = null;

            ISession session = NHibernateSessionManager.Instance.GetSession();
            ITransaction tx = session.BeginTransaction();

            ICriteria criteria = session.CreateCriteria(typeof(FriendListItem));
            criteria.Add(Expression.Eq("FriendListOwner", friendlistowner));

            retVal = criteria.List<FriendListItem>() as List<FriendListItem>;

            tx.Commit();

            return retVal;

        }

        public bool MoneyTransferRequest(LLUUID from, LLUUID to, uint amount)
        {
            //throw new NotImplementedException();
            return false;
        }

        public bool InventoryTransferRequest(LLUUID from, LLUUID to, LLUUID inventory)
        {
            //throw new NotImplementedException();
            return false;
        }

        public string GetVersion()
        {
            return "0.1";
        }

        public string getName()
        {
            return "Hibernate User Dao";
        }

        public void Initialise()
        {

        }

        #endregion

        #region IUserData Members


 
        #endregion
    }
}